// Expected global variables:
/*global $ clipboardData*/

var toolwindowHelpers = {
    
    // A set of types used in the console for different output items
    codeMarkers: {
        perfBrowserTools_DiagnosticsToolWindowsConsoleReady: 23609,
        perfBrowserTools_DiagnosticsToolWindowsDomExplorerReady: 23610,
        perfBrowserTools_DiagnosticsToolWindowsExpandConsoleObjectBegin: 23611,
        perfBrowserTools_DiagnosticsToolWindowsExpandConsoleObjectEnd: 23612,
        perfBrowserTools_DiagnosticsToolWindowsExpandConsoleObjectInteractive: 23613,
        perfBrowserTools_DiagnosticsToolWindowsConsoleEvalBegin: 23614,
        perfBrowserTools_DiagnosticsToolWindowsConsoleEvalEnd: 23615,
        perfBrowserTools_DiagnosticsToolWindowsDataTreeToggleBegin: 23616,
        perfBrowserTools_DiagnosticsToolWindowsDataTreeToggleEnd: 23617,
        perfBrowserTools_DiagnosticsToolWindowsTreeViewToggleBegin: 23618,
        perfBrowserTools_DiagnosticsToolWindowsTreeViewToggleEnd: 23619,
        perfBrowserTools_DiagnosticsToolWindowsDomExplorerRefreshBegin: 23620,
        perfBrowserTools_DiagnosticsToolWindowsDomExplorerRefreshEnd: 23621,
        perfBrowserTools_DiagnosticsToolWindowsDomExplorerAttributeChanged: 23622,
        perfBrowserTools_DiagnosticsToolWindowsDomExplorerTabChanged: 23623
    },
    
    // Number of times to re-send a break mode evaluation message
    maxBreakModeRetries: 5,
    
    // The VSBridge apis
    externalApis: null,
    
    // Communication callbacks
    onMessageCallback: null,
    onAttachCallback: null,
    onDetachCallback: null,
    
    // The current unique id used for identifying callbacks
    uid: 0,
    
    // The communication port that is talking to the remote script engine
    remotePort: null,
    
    // A queue of messages waiting to be sent to the remote side for processing
    pendingMessages: [],
    
    // The current timeout used for batching messages to the remote side
    pendingTimeout: null,
    
    // Queued callbacks waiting to be executed
    callbacks: {},
    
    // Should we be firing code markers to the VSBridge
    areCodeMarkersEnabled: false,
    
    // The state of the program in the debugger
    atBreakpoint: false,

    // Whether we should use VS Evaluation to send messages while at a breakpoint
    useBreakpointEval : false,
    
    initializeToolWindow: function (externalApis, useBreakpointEval, onMessageCallback, onAttachCallback, onDetachCallback, onShowCallback, onBreakCallback, onRunCallback) {
        /// <summary>
        ///     Initializes common functionality of the tool windows
        ///     This includes adding event handlers and styling for buttons and togglebuttons, and
        ///     adding common keyboard navigation functionality
        /// </summary>
        /// <param name="externalApis" type="Object">
        ///     The window.external object that will be used to set up the communication channel
        /// </param>
        /// <param name="onMessageCallback" type="Function">
        ///     The function to call when a message returns from the remote side
        /// </param>
        /// <param name="onAttachCallback" type="Function">
        ///     The function to call when the diagnostics attach to a process
        /// </param>
        /// <param name="onDetachCallback" type="Function">
        ///     The function to call when the diagnostics detach from a process
        /// </param>
        /// <param name="onShowCallback" type="Function" optional="true">
        ///     The function to call when the toolwindow is shown
        /// </param>
        /// <param name="onBreakCallback" type="Function" optional="true">
        ///     The function to call when the debugger goes into break mode
        /// </param>
        /// <param name="onRunCallback" type="Function" optional="true">
        ///     The function to call when the debugger resumes from break mode
        /// </param>
        
        toolwindowHelpers.externalApis = externalApis;
        toolwindowHelpers.useBreakpointEval = useBreakpointEval;
        toolwindowHelpers.onMessageCallback = onMessageCallback;
        toolwindowHelpers.onAttachCallback = onAttachCallback;
        toolwindowHelpers.onDetachCallback = onDetachCallback;
        toolwindowHelpers.onShowCallback = onShowCallback;
        toolwindowHelpers.onBreakCallback = onBreakCallback;
        toolwindowHelpers.onRunCallback = onRunCallback;
        
        // Add the handler that will activate our tool window in VS
        $(document).bind("mousedown", function () { 
            externalApis.vsBridge.notifyOnBrowserActivate(); 
            $(document.body).removeClass("showFocus");
        }, true);
        
        // Prevent the default context menu
        $(document).bind("contextmenu", function () { 
            return false; 
        });
        
        // Prevent the default F5 refresh and shift F10 WPF context menu (the jquery 'contextmenu' event will fire when desired)
        $(document).bind("keydown", function (event) { 
            if (event.keyCode === 116 ||                    
                (event.keyCode === 121 && event.shiftKey)) { // F5(116) and F10(121)
                event.preventDefault();
                event.stopPropagation();
                return false;
            } else if (event.keyCode === 9) { // Tab(9)
                $(document.body).addClass("showFocus");
            }
        });

        // Create focus handlers for css changes
        var focusOut = function () {
            $(document.body).addClass("BPT-ToolWindow-NoFocus");
        };
        var focusIn = function () {
            $(document.body).removeClass("BPT-ToolWindow-NoFocus");
        };
        if (externalApis.vsBridge.getIsToolWindowActive()) {
            focusIn(); // Default to focused
        } else {
            focusOut(); // Default to no focus
        }
        
        // Add the focus handlers
        externalApis.vsBridge.addEventListener("toolwindowShow", toolwindowHelpers.onShow);
        externalApis.vsBridge.addEventListener("toolwindowActivate", focusIn);
        externalApis.vsBridge.addEventListener("toolwindowDeactivate", focusOut);
        externalApis.vsBridge.addEventListener("browserDeactivate", function () {
            // When the user clicks outside the browser but remains in the same tool window
            // we need to remove focus from the current element, so we set it to the
            // body which is tab index -1.
            document.body.setActive();
            $(document.body).removeClass("showFocus");
        });

        
        // Setup the buttons and toggle buttons
        $(".BPT-ToolbarButton").bind("mousedown", function (event) {
            var element = $(this);
            if (!element.hasClass("BPT-ToolbarButton-StateDisabled")) {
                element.addClass("BPT-ToolbarButton-MouseDown");
            } else {
                event.stopImmediatePropagation();
            }
        });
        $(".BPT-ToolbarButton").bind("mouseup", function () {
            $(this).removeClass("BPT-ToolbarButton-MouseDown");
        });
        $(".BPT-ToolbarButton").bind("mouseleave", function () {
            $(this).removeClass("BPT-ToolbarButton-MouseDown BPT-ToolbarButton-MouseHover");
        });
        $(".BPT-ToolbarButton").bind("mouseenter", function (event) {
            var element = $(this);
            if (!element.hasClass("BPT-ToolbarButton-StateDisabled")) {
                element.addClass("BPT-ToolbarButton-MouseHover");
            } else {
                event.preventDefault();
                event.stopImmediatePropagation();
            }
        });
        $(".BPT-ToolbarButton").bind("click keydown", function (event) {
            if (event.type === "click" || event.keyCode === 13 || event.keyCode === 32) { // Enter(13) and Space(32)
                var element = $(this);
                if (!element.hasClass("BPT-ToolbarButton-StateDisabled")) {
                    if (document.activeElement !== element[0]) {
                        element[0].focus();
                    }
                } else {
                    event.preventDefault();
                    event.stopImmediatePropagation();
                }
            }
        });
        $(".BPT-ToolbarToggleButton").bind("click keydown", function (event) {
            if (event.type === "click" || event.keyCode === 13 || event.keyCode === 32) { // Enter(13) and Space(32)
                var element = $(this);
                if (!element.hasClass("BPT-ToolbarButton-StateDisabled")) {
                    if (document.activeElement !== element[0]) {
                        element[0].focus();
                    }
                    element.toggleClass("BPT-ToolbarToggleButton-StateOn");
                } else {
                    event.preventDefault();
                    event.stopImmediatePropagation();
                }
            }
        });
        
        // Setup keyboard navigation
        $(".BPT-TabCycle-Horizontal, .BPT-TabCycle-Vertical").children(".BPT-TabCycle-Item").bind("keydown", function (event) {
            if (($(this).parent().hasClass("BPT-TabCycle-Horizontal") && (event.keyCode === 37 || event.keyCode === 39)) || // Left(37) and Right(39)
                ($(this).parent().hasClass("BPT-TabCycle-Vertical") && (event.keyCode === 38 || event.keyCode === 40))) {   // Up(38) and Down(4)
                var currentElement = $(this);
                var newElement = ((event.keyCode === 37 || event.keyCode === 38) ? // Left(37) or Up(38)
                                    currentElement.prev(".BPT-TabCycle-Item:first") : 
                                    currentElement.next(".BPT-TabCycle-Item:first"));
                
                // Ensure we are moving to a new element
                if (newElement[0]) {
                    newElement.attr("tabindex", "1");
                    newElement.trigger("focus");
                    newElement.trigger("click");
                    currentElement.removeAttr("tabindex");
                }
            }
        });
        $(".BPT-TabCycle-Horizontal, .BPT-TabCycle-Vertical").children(".BPT-TabCycle-Item").bind("mousedown", function (event) {
            var oldElement = $(this).siblings(".BPT-TabCycle-Item[tabindex='1']");
            var newElement = $(this);
            
            // Replace the tab index from the old element, to the new one
            if (newElement[0]) {
                newElement.attr("tabindex", "1");
                newElement.trigger("focus");
                oldElement.removeAttr("tabindex");
            }
        });
        
        toolwindowHelpers.areCodeMarkersEnabled = toolwindowHelpers.externalApis.vsBridge.getAreCodeMarkersEnabled();
        
        // Setup communication channel
        toolwindowHelpers.externalApis.addEventListener("break", toolwindowHelpers.onBreak);
        toolwindowHelpers.externalApis.addEventListener("run", toolwindowHelpers.onRun);
        toolwindowHelpers.externalApis.addEventListener("detach", toolwindowHelpers.onDetach);
        toolwindowHelpers.externalApis.addEventListener("connect", toolwindowHelpers.onConnect);

        if (toolwindowHelpers.externalApis.isAttached) {
            toolwindowHelpers.onAttach();
        } else {
            // Show the diagnostics disabled warning
            $("#warningSection").text(toolwindowHelpers.loadString("DiagnosticsDisabled")).show();
        }
        toolwindowHelpers.externalApis.addEventListener("attach", toolwindowHelpers.onAttach);
    },
    
    onAttach: function () {
        /// <summary>
        ///     The onAttach handler that is called when the diagnostics engine has attached to a process 
        ///     and debugging has started
        /// </summary>
        
        // Hide the diagnostics disabled warning
        $("#warningSection").hide();
        
        toolwindowHelpers.atBreakpoint = toolwindowHelpers.externalApis.vsBridge.getIsAtBreakpoint();
        
        toolwindowHelpers.onAttachCallback();
    },

    onDetach: function () {
        /// <summary>
        ///     The onDetach handler that is called when the diagnostics engine has detached from a process
        ///     and debugging has stopped
        /// </summary>

        // Remove un-needed objects
        toolwindowHelpers.remotePort = null;
        toolwindowHelpers.atBreakpoint = false;
        toolwindowHelpers.callbacks = {};
        toolwindowHelpers.pendingMessages = [];
        toolwindowHelpers.pendingTimeout = null;
        
        toolwindowHelpers.onDetachCallback();
        
        // Show the diagnostics disabled warning
        $("#warningSection").text(toolwindowHelpers.loadString("DiagnosticsDisabled")).show();
    }, 

    onShow: function () {
        /// <summary>
        ///     The onShow handler that is called when the toolwindow has been shown
        /// </summary>    
        
        if (toolwindowHelpers.onShowCallback) {
            toolwindowHelpers.onShowCallback();
        }
    },
    
    onBreak: function () {
        /// <summary>
        ///     The onBreak handler that is called when the diagnostics debugging is paused at a breakpoint
        /// </summary>
        
        // We may have detached and then attached to a non diagnostic enabled debug session,
        // So check that we have a active port connection.
        if (toolwindowHelpers.remotePort) {
            toolwindowHelpers.atBreakpoint = true;
        }
        
        if (toolwindowHelpers.onBreakCallback) {
            toolwindowHelpers.onBreakCallback();
        }
    },

    onRun: function () {
        /// <summary>
        ///     The onRun handler that is called when the diagnostics debugging has resumed from a breakpoint
        /// </summary>
        
        // We may have detached and then attached to a non diagnostic enabled debug session,
        // So check that we have a active port connection.
        if (toolwindowHelpers.remotePort) {
            toolwindowHelpers.atBreakpoint = false;
        }
        
        if (toolwindowHelpers.onRunCallback) {
            toolwindowHelpers.onRunCallback();
        }
    },  

    onConnect: function (port) {
        /// <summary>
        ///     This method is called back when the remote side has connected and is ready for use
        /// </summary>
        /// <param name="port" type="Object">
        ///     The communication object that will be used to post messages
        /// </param>

        toolwindowHelpers.remotePort = port;
        toolwindowHelpers.remotePort.addEventListener("message", toolwindowHelpers.onMessageCallback);
        toolwindowHelpers.atBreakpoint = toolwindowHelpers.externalApis.vsBridge.getIsAtBreakpoint();
    },

    registerErrorComponent: function (component, errorDisplayHandler) {
        /// <summary>
        ///     Stores the component name and error handler function for non-fatal
        ///     error reporting    
        /// </summary>
        /// <param name="component" type="String">
        ///     The identifying name of the component
        /// </param>
        /// <param name="errorDisplayHandler" type="Function">
        ///     The function that should be called to display an error message to the
        ///     user
        /// </param>   
        
        window.errorComponent = component;
        window.errorDisplayHandler = errorDisplayHandler;
    },
    
    registerThemeChange: function (externalApis, cssFiles) {
        /// <summary>
        ///     Adds an event listener to the themeChange event in Visual Studio and 
        ///     processes the css files when a theme change occurs    
        /// </summary>
        /// <param name="externalApis" type="Object">
        ///     The window.external object that will be used to set up the communication channel
        /// </param>
        /// <param name="cssFiles" type="Array">
        ///     Array of strings containing the name of the css files required for the toolwindow
        /// </param>
     
        function onThemeChange() {
            var cssThemeFiles = ["toolwindow.css", "datatree.css", "htmltree.css"].concat(cssFiles);
            for (var i = 0; i < cssThemeFiles.length; i++) {

                var id = cssThemeFiles[i];
                var contents = externalApis.vsBridge.loadCssFile(cssThemeFiles[i], i < 3);

                // Remove any previous style with this id
                var oldStyle = document.getElementById(id);
                if (oldStyle) {
                    document.head.removeChild(oldStyle);
                }

                // Add the new ones
                var styleNode = document.createElement("style");
                styleNode.id = id;
                styleNode.type = "text/css";
                styleNode.innerHTML = contents;
                document.head.appendChild(styleNode);
            }
            
            // Check the tree expand icons
            var trees = $(".BPT-HtmlTree, .BPT-DataTree, .BPT-DataTree-ScrollContainer, .BPT-Toolbar");
            for (var j = 0; j < trees.length; j++) {
                var element = $(trees[j]);
                var useDarkTheme = toolwindowHelpers.isDarkThemeBackground(element);
                if (useDarkTheme) {
                    element.addClass("BPT-Tree-DarkTheme");
                } else {
                    element.removeClass("BPT-Tree-DarkTheme");
                }
            }
        }
        
        // Ensure the WebOC version is supported
        if (document.documentMode >= 10 || (window.parent.getExternalObj && document.documentMode >= 9)) {
            externalApis.vsBridge.addEventListener("themeChange", onThemeChange);
            
            // Fire an initial theme change event
            if (!window.parent.getExternalObj) {
                onThemeChange();
            }
        } else {
            // This browser version is not supported
            externalApis.vsBridge.notifyUnsupportedBrowser(document.documentMode);
            window.navigate("about:blank");
        }
    },
    
    loadString: function (resourceId, params) {
        /// <summary>
        ///     Gets a localized string for the resourceId
        /// </summary>
        /// <param name="resourceId" type="String">
        ///     The resource identifier of the string to retrieve
        /// </param>
        /// <param name="params" type="Array" optional="true">
        ///     Optional array of objects to be used in any format specifiers in the string
        /// </param>
        /// <returns type="String">
        ///     The loaded localized string
        /// </returns>    

        if (params !== undefined) {
            // Use the format function
            return toolwindowHelpers.externalApis.vsBridge.loadFormattedString(resourceId, params);
        } else {
            // Use no formatting
            return toolwindowHelpers.externalApis.vsBridge.loadString(resourceId);
        }
    },
    
    codeMarker: function (codeMarker) {
        /// <summary>
        ///     Fire a VS perf code marker with a specified identifier
        /// </summary>
        /// <param name="codeMarker" type="Number">
        ///     The value of the code marker
        /// </param>
        
        if (toolwindowHelpers.areCodeMarkersEnabled) {
            toolwindowHelpers.externalApis.vsBridge.fireCodeMarker(codeMarker);
        }
    },

    executeBreakModeCommand: function (remoteFunction, id, input, callback) {
        /// <summary>
        ///     Executes a command at breakmode using VS instead of the remote code,
        ///     This is used by the console to evaluate input in the context of the current breakpoint.
        /// </summary>
        /// <param name="remoteFunction" type="String">
        ///     The function to call on the remote side 
        /// </param>
        /// <param name="id" type="String">
        ///     The id to give to this command so that it can be identifed on the remote side
        /// </param>        
        /// <param name="input" type="String">
        ///     The input to execute
        /// </param>
        /// <param name="callback" type="Function">
        ///     The function to execute when the result is returned from the remote side
        /// </param>
        
        var uidString = toolwindowHelpers.getUid();
        toolwindowHelpers.callbacks[uidString] = { synced: true, callback: callback || $.noop };
    
        var sendBreakCommand = function (remoteFunction, id, uidString, input) {
            if (toolwindowHelpers.atBreakpoint) {
                // Send the message using the break mode command evaluator
                var succeeded = toolwindowHelpers.externalApis.vsBridge.executeBreakModeCommand(remoteFunction + ":" + id + ":" + uidString, input);
                if (!succeeded) {
                    // We failed to send the break mode command because it is no longer in break mode,
                    // so send the message via the run time remote port.
                    if (toolwindowHelpers.remotePort) {
                        var jsonObj = {
                            uid: uidString,
                            command: remoteFunction,
                            args: [id, input]
                        };
                        var message = JSON.stringify([jsonObj]);
                        toolwindowHelpers.remotePort.postMessage(message);
                    }
                }
            }
        };
        
        setTimeout(function () {
            sendBreakCommand(remoteFunction, id, uidString, input);
        }, 0);
    },
    
    scrollIntoView: function (element, scrollContainer) {
        /// <summary>
        ///     Scrolls an element into view in a scroll container if it is currently outside of the view
        /// </summary>
        /// <param name="element" type="Object">
        ///     The DOM element that should be scrolled into the view
        /// </param>
        /// <param name="scrollContainer" type="Object">
        ///     The DOM element that has scrollbars and has the element being scrolled as a decendant
        /// </param>        
        /// <returns type="Boolean">
        ///     True if the view was scrolled, False if the element was already in the view and did not need scrolling
        /// </returns>
        
        // Ensure we have a valid element to scroll
        if (element) {
            var topOfPage = scrollContainer.scrollTop;
            var heightOfPage = scrollContainer.clientHeight;
            var elementTop = 0;
            var elementHeight = 0;

            var currentElement = element;
            while (currentElement && currentElement !== scrollContainer) {
                elementTop += currentElement.offsetTop;
                currentElement = currentElement.offsetParent;
            }
            elementHeight = element.offsetHeight;
            
            var alignPosition;
            if ((topOfPage + heightOfPage) < (elementTop + elementHeight)) {
                alignPosition = "bottom";
            } else if (elementTop < topOfPage) {
                alignPosition = "top";
            }
            
            if (alignPosition) {
                var temp = $("<div style='position:absolute; top:" + elementTop + "px; left: 0; height:" + elementHeight + "px'></div>");
                $(scrollContainer).append(temp);
                temp[0].scrollIntoView(alignPosition === "top");
                temp.remove();
                return true;           
            }
        }
        
        return false;

    },
    
    getSortedObjectProperties: function (objectToSort) {
        /// <summary>
        ///     Sorts an object's property names alphabetically and returns an array of the sorted names
        /// </summary>
        /// <param name="objectToSort" type="Object">
        ///     The javascript object that contains the properties that need to be sorted
        /// </param>
        /// <returns type="Array">
        ///     An array of the sorted property names that can be used as a list of sorted keys into the real object
        /// </returns>
        
        // Sort the property names for display
        var sortedPropNames = [];
        for (var propName in objectToSort) {
            sortedPropNames.push(propName);
        }

        sortedPropNames.sort(toolwindowHelpers.naturalSort);
        
        return sortedPropNames;
    },
    
    getSortedArrayProperties: function (arrayToSort, key, highPriorityValue) {
        /// <summary>
        ///     Sorts an array of objects on a key property names alphabetically and returns an array of the sorted indicies
        /// </summary>
        /// <param name="arrayToSort" type="Array">
        ///     The javascript array that contains the objects that need to be sorted
        /// </param>
        /// <param name="key" type="String">
        ///     The name of the property to sort the array by
        /// </param>
        /// <param name="highPriorityValue" type="String" optional="true">
        ///     Optional parameter to specify a value that should be treated with highest priority in the sort
        /// </param>           
        /// <returns type="Array">
        ///     An array of the sorted indicies that can be used as a list of sorted keys into the real array
        /// </returns>
        
        // Sort the property names for display
        var i;
        var sortedProps = [];
        for (i = 0; i < arrayToSort.length; i++) {
            sortedProps.push({property: arrayToSort[i][key], realIndex: i});
        }
       
        sortedProps.sort(function (a, b) {
            if (highPriorityValue) {
                if (a.property === highPriorityValue && b.property === highPriorityValue) {
                    return 0;
                } else if (a.property === highPriorityValue) {
                    return -1;
                } else if (b.property === highPriorityValue) {
                    return 1;
                }
            }
            return toolwindowHelpers.naturalSort(a.property, b.property);
        });
        
        var sortedList = [];
        for (i = 0; i < sortedProps.length; i++) {
            sortedList.push(sortedProps[i].realIndex);
        }
        
        return sortedList;
    },  
    
    naturalSort: function (a, b) {
        /// <summary>
        ///     Sorts two objects as strings alphabetically and returns a number representing the order
        /// </summary>
        /// <param name="a" type="Object">
        ///     The first string object to compare
        /// </param>
        /// <param name="b" type="Object">
        ///     The second string object to compare
        /// </param>        
        /// <returns type="Number">
        ///     A number representing the sort order
        ///     a &gt; b = 1
        ///     a &lt; b = -1
        ///     a == b = 0
        /// </returns>
        
        // Regular Expression to pick groups of either digits or non digits (eg. 11bc34 - will return [11, bc, 34])
        var regexSortGroup = /(\d+)|(\D+)/g;
        
        // Convert to case insensitive strings and identify the sort groups
        var aGroups = String(a).toLowerCase().match(regexSortGroup);
        var bGroups = String(b).toLowerCase().match(regexSortGroup);

        // Loop through each group
        while (aGroups.length > 0 && bGroups.length > 0) {
            // Take the first group of each string
            var aFront = aGroups.shift();
            var bFront = bGroups.shift();
            
            // Check for digits
            var aAsDigit = parseInt(aFront, 10);
            var bAsDigit = parseInt(bFront, 10);
            
            if (isNaN(aAsDigit) && isNaN(bAsDigit)) {
                // Compare as string characters
                if (aFront !== bFront) {
                    // Chars not the same, so just return the sort value
                    return (aFront > bFront ? 1 : -1);
                }
            } else if (isNaN(aAsDigit)) {
                // Letters come after numbers
                return 1;
            } else if (isNaN(bAsDigit)) {
                // Numbers come before letters
                return -1;
            } else {
                // Compare as numbers
                if (aAsDigit !== bAsDigit) {
                    // Numbers not the same, so just return the sort value
                    return (aAsDigit - bAsDigit);
                }
            }
        }
        
        // If we get here, we know all the groups checked were identical,
        // So we can return the length difference as the sort value.
        return aGroups.length - bGroups.length;
    },
    
    formatMultilineString: function (multilineString) {
        /// <summary>
        ///     Formats a multiline string by trimming whitespace and removing any empty lines
        /// </summary>
        /// <param name="multilineString" type="String">
        ///     The string to format
        /// </param>
        /// <returns type="String">
        ///     The formatted string
        /// </returns>
        
        if (!multilineString) {
            return "";
        }
        
        var text = "";
        
        var finalLines = [];
        // Split into lines, then process each one
        var lines = multilineString.split(/[\r\n]+/);
        for (var lineIndex = 0; lineIndex < lines.length; lineIndex++) {
            if ($.trim(lines[lineIndex]) !== "") {
                finalLines.push(lines[lineIndex]);
            }
        }
        
        // Join up the lines into html with line breaks
        text = finalLines.join("\n");
        
        return text;
    },

    createShortenedUrlText: function (url) {
        /// <summary>
        ///     Returns a short form of the URL for use in displaying file links to the user.  Adapted
        ///     from F12 toolbar code, this method removes any trailing query string or anchor location 
        ///     and attempts to get the last file or directory following a '/'.  
        /// </summary>
        /// <param name="url" type="String">
        ///     The url to shorten.
        /// </param>
        /// <returns type="String">
        ///     A shortened version of the string
        /// </returns>
        var shortenedText = url;
    
        // Remove a query string if any
        var indexOfHash = shortenedText.indexOf("#");
        var indexOfQuestionMark = shortenedText.indexOf("?");
        var index = -1;
        if (indexOfHash > -1 && indexOfQuestionMark > -1) {
            index = Math.min(indexOfHash, indexOfQuestionMark);
        } else if (indexOfHash > -1) {
            index = indexOfHash;
        } else if (indexOfQuestionMark > -1) {
            index = indexOfQuestionMark;
        }
    
        if (index > -1) {
            shortenedText = shortenedText.substring(0, index);
        }
    
        index = shortenedText.lastIndexOf("/");
    
        // While the last character is '/', truncate it and find the next / or the start of the string
        while (index === (shortenedText.length - 1) ) {
            // Remove last '/'
            shortenedText = shortenedText.substring(0, shortenedText.length - 1);
            index = shortenedText.lastIndexOf("/");
        }
    
        if (index > -1) {
            shortenedText = shortenedText.substring(index + 1);
        }
    
        return shortenedText;
    },

    createLinkDivText: function (link, styles, dontGenerateTooltip) {
        /// <summary>
        ///     Creates a DOM div element string for a link that opens a document in VS
        /// </summary>
        /// <param name="link" type="String">
        ///     The href to turn into a link div.
        /// </param>
        /// <param name="styles" type="String" optional="true">
        ///     A ' ' separated list of styles to add to the div.
        /// </param>
        /// <param name="dontGenerateTooltip" type="Boolean" optional="true">
        ///     No tooltip will be added if set to true.
        /// </param>
        /// <returns type="String">
        ///     The link string of the form <div class='[styles string] BPT-FileLink' data-linkUrl="[link.url]" data-linkSearch="[link.search]" data-linkLine='[link.line]' data-linkCol='[link.column]' tabindex='1'>[shortened link.url]</div>
        /// </returns>
        
        var linker = "";
        if (link && link.url) {
            // Create the url and text
            var url = "\"" + link.url.replace(/\\"/g, "\"") + "\"";
            var linkText = toolwindowHelpers.createShortenedUrlText(link.url);
            
            // Create the search term
            var search = "";
            if (link.search) {
                search = "\"" + link.search.replace(/\\"/g, "\"") + "\"";
            }
            
            linkText = linkText.replace(/&/g, "&amp;").replace(/>/g, "&gt;").replace(/</g, "&lt;");
            linker = "<div class='" + styles + " BPT-FileLink' data-linkUrl=" + url; 
            if (search) {
               linker += " data-linkSearch=" + search;
            }
            if (link.line) {
                linker += " data-linkLine='" + link.line + "'";
            }
            if (link.column) {
                linker += " data-linkCol='" + link.column + "'";
            }
            if (!dontGenerateTooltip) {
                linker += " title='" + linkText + "'";
            }
            linker += ">" + linkText + "</div>";
        }

        return linker;
    },

    htmlEscape: function (htmlString) {
        /// <summary>
        ///     Escapes a string so that it can be safely displayed in html. 
        /// </summary>
        /// <param name="htmlString" type="String">
        ///     The javascript string that is to be HTML escaped
        /// </param>
        /// <returns type="String">
        ///     The escaped string 
        /// </returns>

        // Ensure we have a string to escape
        if ((typeof htmlString) !== "string") {
            if (htmlString === null || htmlString === undefined) {
                return "";
            }
            htmlString = "" + htmlString;
        }
        
        // Speed up the html escape by using a regular expression to encode html characters
        return htmlString.replace(/&/g, "&amp;").replace(/>/g, "&gt;").replace(/</g, "&lt;").replace(/"/g, '&quot;').replace(/'/g, '&#39;');
    },
    
    copySelectedTextToClipboard: function () {
        /// <summary>
        ///     Gets the highlighted text in the document, compacts multiline text by converting multiple \r\n's to a single one, and then copies the text to the clipboard
        /// </summary>
        /// <returns type="Boolean">
        ///     True if any text was copied, false otherwise
        /// </returns>
        
        var selectedText = document.selection.createRange().text;
        if (selectedText) {
            // Replace multiple white space lines with a single one
            var compactText = selectedText.replace(/[\r\n]+/g, "\r\n");
            // Copy to the clipboard
            clipboardData.setData("Text", compactText);
            return true;
        }
        
        return false;
    },
    
    isDarkThemeBackground: function (element) {
        /// <summary>
        ///     Checks the element's background color to see if it is being displayed in the dark theme
        /// </summary>
        /// <param name="element" type="Object">
        ///     The JQuery element to check the background for
        /// </param>        
        /// <returns type="Boolean">
        ///     True if the background color indicates the dark theme, False if it is light
        /// </returns>

        if (element) {
            var backgroundColor;
            while ((!backgroundColor || backgroundColor === "transparent") && element && element.length > 0) {
                backgroundColor = element.css("background-color");
                
                element = element.parent();
            }
            
            var rgbParts = backgroundColor.match(/^rgb\((\d+),\s*(\d+),\s*(\d+)\)$/);
            if (rgbParts && rgbParts.length === 4) {
                
                // Brightness determined by W3C formula
                var brightness = ((rgbParts[1] * 299) + (rgbParts[2] * 587) + (rgbParts[3] * 114)) / 1000;
                
                return (brightness < 127);
            }
        }
        
        // Default to using light theme
        return false;
    },
    
    getUid: function () {
        /// <summary>
        ///     This function returns a new unique identifier string for the dom explorer window
        /// </summary>
        /// <returns type="String">
        ///     A string representing the unique id
        /// </returns>
        
        return "uid" + (toolwindowHelpers.uid++).toString(36);
    }    
};

// Create the proxy for our communications
window.callProxy = function callProxy(command, args, callback) {
    /// <summary>
    ///     Sends a message to the remote side to execute an async function and optionally
    ///     return a result
    /// </summary>
    /// <param name="command" type="String">
    ///     The remote function command to call
    /// </param>
    /// <param name="args" type="Array" optional="true">
    ///     An optional array of arguments to pass to the remote function
    /// </param>
    /// <param name="callback" type="Function" optional="true">
    ///     An optional callback function to trigger when the async result is recieved
    /// </param>
        
    // Generate a unique id to track this transaction
    var uidString = toolwindowHelpers.getUid();
    
    if (callback) {
        // Only create a callback object if there is something that will be called back
        toolwindowHelpers.callbacks[uidString] = { synced: true, callback: callback || $.noop };
    }
    
    var newArgs = $.map(args || [], function (arg) {
        if (typeof (arg) === "function") {
            var callbackuid = toolwindowHelpers.getUid();
            toolwindowHelpers.callbacks[callbackuid] = { synced: false, callback: arg };

            return { uid: callbackuid,
                type: "callback"
            };

        } else {
            return arg;
        }
    });
    var jsonObj = {
        uid: uidString,
        command: command,
        args: newArgs
    };

    var sendMessageToRemote = function (message) {
        // Ensure that we have a port to send the message to as we may have disconnected
        if (toolwindowHelpers.remotePort) {
            // Send the message via the run time remote port
            toolwindowHelpers.remotePort.postMessage(message);
        }
    };
            
    toolwindowHelpers.pendingMessages.push(jsonObj);
    if (!toolwindowHelpers.pendingTimeout) {
        toolwindowHelpers.pendingTimeout = setTimeout(function () {
            var message = JSON.stringify(toolwindowHelpers.pendingMessages);
            toolwindowHelpers.pendingMessages = [];
            toolwindowHelpers.pendingTimeout = null;
            sendMessageToRemote(message);
        }, 0);
    }
};

window.callProxy.fireCallbacks = function (data) {
    var msgs = JSON.parse(data);
    for (var i = 0; i < msgs.length; i++) {
        var obj = msgs[i];
        if (toolwindowHelpers.callbacks[obj.uid]) {
            if (obj.args !== undefined) {
                toolwindowHelpers.callbacks[obj.uid].callback.apply(this, obj.args);
            }
            
            // Ensure the callback still exists incase it has already been removed by cleanup in the previous call
            if (toolwindowHelpers.callbacks[obj.uid] && toolwindowHelpers.callbacks[obj.uid].synced) {
                delete toolwindowHelpers.callbacks[obj.uid];
            }
        } else if (obj.uid === "scriptError") {
            // Fire the script error handler
            window.reportError(obj.args[0].message, obj.args[0].file, obj.args[0].line, obj.args[0].additionalInfo);
        }
    }
};

window.reportError = function (message, file, line, additionalInfo) {
    /// <summary>
    ///     Handles JavaScript errors in the toolwindows by reporting them as non-fatal errors
    /// </summary>
    /// <param name="message" type="String">
    ///     The error message
    /// </param>
    /// <param name="file" type="String">
    ///     The file in which the error occurred
    /// </param>
    /// <param name="line" type="Number">
    ///     The line on which the error occurred
    /// </param>
    /// <param name="additionalInfo" type="String">
    ///     Any additional information about the error such as callstack
    /// </param>
    
    var externalObj;
    if (window.parent.getExternalObj) {
        // Hosted in an IFRAME, so get the external object from there
        externalObj = window.parent.getExternalObj();
    } else if (window.external) { 
        // Hosted in Visual Studio
        externalObj = window.external;
    }
    
    // Add additional vs side information
    var info = [];
    try {
        info.push(additionalInfo);
        info.push("atBreakpoint: " + toolwindowHelpers.atBreakpoint);
        info.push("Stored Callbacks: " + JSON.stringify(toolwindowHelpers.callbacks));
        info.push("Pending Messages: " + JSON.stringify(toolwindowHelpers.pendingMessages));
    } catch (ex3) {
        // Fail gracefully
    }
    var finalInfo = info.join("\r\n\r\n");
    
    if (externalObj) {
        // Code in this function is commented out to stop the error from being displayed to the user for release,
        // If we need this this functionality again, the code can be un-commented.
        
        // Store the script error for later use
        ////window.lastScriptError = {message: message, file: file, line: line, additionalInfo: finalInfo};
        
        // Report the NFE to the watson server
        var component = (window.errorComponent ? window.errorComponent : "Common");
        externalObj.reportNonFatalError(component, message, file, line, finalInfo);
        
        // Display a warning message to the user
        ////if (window.errorDisplayHandler) {
        ////    window.errorDisplayHandler(message, file, line, finalInfo);
        ////}
    }
 
};

window.onerror = function (message, file, line) {
    /// <summary>
    ///     Handles JavaScript errors in the toolwindows by reporting them as non-fatal errors
    /// </summary>
    /// <param name="message" type="String">
    ///     The error message
    /// </param>
    /// <param name="file" type="String">
    ///     The file in which the error occurred
    /// </param>
    /// <param name="line" type="Number">
    ///     The line on which the error occurred
    /// </param>
    /// <returns type="Boolean">
    ///     Returns true to mark the error as handled, False to display the default error dialog
    /// </returns>

    // The maximum callstack size to collect
    var maxStackSize = 10;

    var getArgumentString = function (argument) {
        /// <summary>
        ///     Gets a string representing the value of the passed in argument.
        ///     This supliments the built in typeof function by calculating the type of certain objects such as
        ///     array, date, and regex
        /// </summary>    
        /// <param name="argument" type="Object">
        ///     The argument to get the value of
        /// </param>
        /// <returns type="String">
        ///     A string representing the value of this argument
        /// </returns>            
        /// <disable>JS3053.IncorrectNumberOfArguments,JS2005.UseShortFormInitializations</disable>
        var type = (typeof argument);

        // Check for undefined
        if (argument === undefined) {
            type = "undefined";
        } else {
            // Check for object type
            if (type === "object") {
                if (argument) {
                    if (typeof argument.length === "number" && typeof argument.propertyIsEnumerable === "function" && !(argument.propertyIsEnumerable("length")) && typeof argument.splice === "function") {
                        type = "array";
                    }
                    try {
                        if (argument.constructor === (new Array()).constructor) {
                            type = "array";
                        } else if (argument.constructor === (new Date()).constructor) {
                            type = "date";
                        } else if (argument.constructor === (new RegExp()).constructor) {
                            type = "regex";
                        }
                    } catch (e) {
                        // This object is not accessible
                    }
                } else {
                    type = "null";
                }
                type = "object";
            }
        }
        
        switch (type) {
            case "boolean":
                return argument;

            case "date":
                return "[date] " + argument;

            case "function":
                return "" + argument;

            case "null":
                return "null";

            case "number":
                return argument;

            case "regex":
                return "[regex] " + argument;

            case "string":
                return "\"" + argument + "\"";

            case "undefined":
                return "undefined";
                
            case "htmlElement":
            // FALLTHROUGH
            case "array":
            // FALLTHROUGH
            case "object":
                return JSON.stringify(argument);
        }

    };

    // Generate the callstack information
    var callstack = [];
    try {
        var currentFunction = arguments.callee;
        var functionText, functionName, match, args, stringifiedArgs;
        
        // Loop up the caller chain
        while (currentFunction && callstack.length < maxStackSize) {
            
            // Set default values
            functionName = "unknown";
            stringifiedArgs = [];
            
            try {
                // Get the function name
                functionText = currentFunction.toString() || "";
                match = functionText.match(/function\s*([\w\-$]+)?\s*\(/i);
                functionName = (match.length >= 2 ? match[1] || "anonymous" : "anonymous");

                // Get the arguments
                if (currentFunction["arguments"]) {
                    args = currentFunction["arguments"];
                    for (var i = 0; i < args.length; i++) {
                        stringifiedArgs.push(getArgumentString(args[i]));
                    }
                }                       
            } catch (ex) {
                // Fail gracefully
            }
            
            // Add this info to the callstack
            callstack.push(functionName + " (" + stringifiedArgs.join(", ") + ")");
            
            // Walk up the stack
            currentFunction = currentFunction.caller;
        }
    } catch (ex2) {
        // Fail gracefully
    }
    
    // Populate the additional information
    var info = [];
    try {
        info.push("Callstack:\r\n" + callstack.join("\r\n"));
    } catch (ex3) {
        // Fail gracefully
    }
    
    window.reportError(message, file, line, info);
    return true;
};


// SIG // Begin signature block
// SIG // MIIakQYJKoZIhvcNAQcCoIIagjCCGn4CAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFI8dUtpshWyf
// SIG // CrJavGyb3+yTuECxoIIVeTCCBLowggOioAMCAQICCmEC
// SIG // kkoAAAAAACAwDQYJKoZIhvcNAQEFBQAwdzELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWljcm9zb2Z0IFRp
// SIG // bWUtU3RhbXAgUENBMB4XDTEyMDEwOTIyMjU1OVoXDTEz
// SIG // MDQwOTIyMjU1OVowgbMxCzAJBgNVBAYTAlVTMRMwEQYD
// SIG // VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25k
// SIG // MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24x
// SIG // DTALBgNVBAsTBE1PUFIxJzAlBgNVBAsTHm5DaXBoZXIg
// SIG // RFNFIEVTTjpCOEVDLTMwQTQtNzE0NDElMCMGA1UEAxMc
// SIG // TWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZTCCASIw
// SIG // DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAM1jw/ei
// SIG // tUfZ+TmUU6xrj6Z5OCH00W49FTgWwXMsmY/74Dxb4aJM
// SIG // i7Kri7TySse5k1DRJvWHU7B6dfNHDxcrZyxk62DnSozg
// SIG // i17EVmk3OioEXRcByL+pt9PJq6ORqIHjPy232OTEeAB5
// SIG // Oc/9x2TiIxJ4ngx2J0mPmqwOdOMGVVVJyO2hfHBFYX6y
// SIG // cRYe4cFBudLSMulSJPM2UATX3W88SdUL1HZA/GVlE36V
// SIG // UTrV/7iap1drSxXlN1gf3AANxa7q34FH+fBSrubPWqzg
// SIG // FEqmcZSA+v2wIzBg6YNgrA4kHv8R8uelVWKV7p9/ninW
// SIG // zUsKdoPwQwTfBkkg8lNaRLBRejkCAwEAAaOCAQkwggEF
// SIG // MB0GA1UdDgQWBBTNGaxhTZRnK/avlHVZ2/BYAIOhOjAf
// SIG // BgNVHSMEGDAWgBQjNPjZUkZwCu1A+3b7syuwwzWzDzBU
// SIG // BgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vY3JsLm1pY3Jv
// SIG // c29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNyb3Nv
// SIG // ZnRUaW1lU3RhbXBQQ0EuY3JsMFgGCCsGAQUFBwEBBEww
// SIG // SjBIBggrBgEFBQcwAoY8aHR0cDovL3d3dy5taWNyb3Nv
// SIG // ZnQuY29tL3BraS9jZXJ0cy9NaWNyb3NvZnRUaW1lU3Rh
// SIG // bXBQQ0EuY3J0MBMGA1UdJQQMMAoGCCsGAQUFBwMIMA0G
// SIG // CSqGSIb3DQEBBQUAA4IBAQBRHNbfNh3cgLwCp8aZ3xbI
// SIG // kAZpFZoyufNkENKK82IpG3mPymCps13E5BYtNYxEm/H0
// SIG // XGGkQa6ai7pQ0Wp5arNijJ1NUVALqY7Uv6IQwEfVTnVS
// SIG // iR4/lmqPLkAUBnLuP3BZkl2F7YOZ+oKEnuQDASETqyfW
// SIG // zHFJ5dod/288CU7VjWboDMl/7jEUAjdfe2nsiT5FfyVE
// SIG // 5x8a1sUaw0rk4fGEmOdP+amYpxhG7IRs7KkDCv18elId
// SIG // nGukqA+YkqSSeFwreON9ssfZtnB931tzU7+q1GZQS/DJ
// SIG // O5WF5cFKZZ0lWFC7IFSReTobB1xqVyivMcef58Md7kf9
// SIG // J9d/z3TcZcU/MIIE7DCCA9SgAwIBAgITMwAAALARrwqL
// SIG // 0Duf3QABAAAAsDANBgkqhkiG9w0BAQUFADB5MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQTAeFw0xMzAxMjQyMjMzMzla
// SIG // Fw0xNDA0MjQyMjMzMzlaMIGDMQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYDVQQDExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEB
// SIG // AQUAA4IBDwAwggEKAoIBAQDor1yiIA34KHy8BXt/re7r
// SIG // dqwoUz8620B9s44z5lc/pVEVNFSlz7SLqT+oN+EtUO01
// SIG // Fk7vTXrbE3aIsCzwWVyp6+HXKXXkG4Unm/P4LZ5BNisL
// SIG // QPu+O7q5XHWTFlJLyjPFN7Dz636o9UEVXAhlHSE38Cy6
// SIG // IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpOoPXJCiHiquMB
// SIG // Nkf9L4JqgQP1qTXclFed+0vUDoLbOI8S/uPWenSIZOFi
// SIG // xCUuKq6dGB8OHrbCryS0DlC83hyTXEmmebW22875cHso
// SIG // AYS4KinPv6kFBeHgD3FN/a1cI4Mp68fFSsjoJ4TTfsZD
// SIG // C5UABbFPZXHFAgMBAAGjggFgMIIBXDATBgNVHSUEDDAK
// SIG // BggrBgEFBQcDAzAdBgNVHQ4EFgQUWXGmWjNN2pgHgP+E
// SIG // Hr6H+XIyQfIwUQYDVR0RBEowSKRGMEQxDTALBgNVBAsT
// SIG // BE1PUFIxMzAxBgNVBAUTKjMxNTk1KzRmYWYwYjcxLWFk
// SIG // MzctNGFhMy1hNjcxLTc2YmMwNTIzNDRhZDAfBgNVHSME
// SIG // GDAWgBTLEejK0rQWWAHJNy4zFha5TJoKHzBWBgNVHR8E
// SIG // TzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NybC9wcm9kdWN0cy9NaWNDb2RTaWdQQ0Ff
// SIG // MDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUHAQEETjBMMEoG
// SIG // CCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NlcnRzL01pY0NvZFNpZ1BDQV8wOC0zMS0y
// SIG // MDEwLmNydDANBgkqhkiG9w0BAQUFAAOCAQEAMdduKhJX
// SIG // M4HVncbr+TrURE0Inu5e32pbt3nPApy8dmiekKGcC8N/
// SIG // oozxTbqVOfsN4OGb9F0kDxuNiBU6fNutzrPJbLo5LEV9
// SIG // JBFUJjANDf9H6gMH5eRmXSx7nR2pEPocsHTyT2lrnqkk
// SIG // hNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2yRPnwPJNtQtjo
// SIG // fOYXoJtoaAko+QKX7xEDumdSrcHps3Om0mPNSuI+5PNO
// SIG // /f+h4LsCEztdIN5VP6OukEAxOHUoXgSpRm3m9Xp5QL0f
// SIG // zehF1a7iXT71dcfmZmNgzNWahIeNJDD37zTQYx2xQmdK
// SIG // Dku/Og7vtpU6pzjkJZIIpohmgjCCBbwwggOkoAMCAQIC
// SIG // CmEzJhoAAAAAADEwDQYJKoZIhvcNAQEFBQAwXzETMBEG
// SIG // CgmSJomT8ixkARkWA2NvbTEZMBcGCgmSJomT8ixkARkW
// SIG // CW1pY3Jvc29mdDEtMCsGA1UEAxMkTWljcm9zb2Z0IFJv
// SIG // b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5MB4XDTEwMDgz
// SIG // MTIyMTkzMloXDTIwMDgzMTIyMjkzMloweTELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEjMCEGA1UEAxMaTWljcm9zb2Z0IENv
// SIG // ZGUgU2lnbmluZyBQQ0EwggEiMA0GCSqGSIb3DQEBAQUA
// SIG // A4IBDwAwggEKAoIBAQCycllcGTBkvx2aYCAgQpl2U2w+
// SIG // G9ZvzMvx6mv+lxYQ4N86dIMaty+gMuz/3sJCTiPVcgDb
// SIG // NVcKicquIEn08GisTUuNpb15S3GbRwfa/SXfnXWIz6pz
// SIG // RH/XgdvzvfI2pMlcRdyvrT3gKGiXGqelcnNW8ReU5P01
// SIG // lHKg1nZfHndFg4U4FtBzWwW6Z1KNpbJpL9oZC/6SdCni
// SIG // di9U3RQwWfjSjWL9y8lfRjFQuScT5EAwz3IpECgixzdO
// SIG // PaAyPZDNoTgGhVxOVoIoKgUyt0vXT2Pn0i1i8UU956wI
// SIG // APZGoZ7RW4wmU+h6qkryRs83PDietHdcpReejcsRj1Y8
// SIG // wawJXwPTAgMBAAGjggFeMIIBWjAPBgNVHRMBAf8EBTAD
// SIG // AQH/MB0GA1UdDgQWBBTLEejK0rQWWAHJNy4zFha5TJoK
// SIG // HzALBgNVHQ8EBAMCAYYwEgYJKwYBBAGCNxUBBAUCAwEA
// SIG // ATAjBgkrBgEEAYI3FQIEFgQU/dExTtMmipXhmGA7qDFv
// SIG // pjy82C0wGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEw
// SIG // HwYDVR0jBBgwFoAUDqyCYEBWJ5flJRP8KuEKU5VZ5KQw
// SIG // UAYDVR0fBEkwRzBFoEOgQYY/aHR0cDovL2NybC5taWNy
// SIG // b3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvbWljcm9z
// SIG // b2Z0cm9vdGNlcnQuY3JsMFQGCCsGAQUFBwEBBEgwRjBE
// SIG // BggrBgEFBQcwAoY4aHR0cDovL3d3dy5taWNyb3NvZnQu
// SIG // Y29tL3BraS9jZXJ0cy9NaWNyb3NvZnRSb290Q2VydC5j
// SIG // cnQwDQYJKoZIhvcNAQEFBQADggIBAFk5Pn8mRq/rb0Cx
// SIG // MrVq6w4vbqhJ9+tfde1MOy3XQ60L/svpLTGjI8x8UJiA
// SIG // IV2sPS9MuqKoVpzjcLu4tPh5tUly9z7qQX/K4QwXacul
// SIG // nCAt+gtQxFbNLeNK0rxw56gNogOlVuC4iktX8pVCnPHz
// SIG // 7+7jhh80PLhWmvBTI4UqpIIck+KUBx3y4k74jKHK6BOl
// SIG // kU7IG9KPcpUqcW2bGvgc8FPWZ8wi/1wdzaKMvSeyeWNW
// SIG // RKJRzfnpo1hW3ZsCRUQvX/TartSCMm78pJUT5Otp56mi
// SIG // LL7IKxAOZY6Z2/Wi+hImCWU4lPF6H0q70eFW6NB4lhhc
// SIG // yTUWX92THUmOLb6tNEQc7hAVGgBd3TVbIc6YxwnuhQ6M
// SIG // T20OE049fClInHLR82zKwexwo1eSV32UjaAbSANa98+j
// SIG // Zwp0pTbtLS8XyOZyNxL0b7E8Z4L5UrKNMxZlHg6K3RDe
// SIG // ZPRvzkbU0xfpecQEtNP7LN8fip6sCvsTJ0Ct5PnhqX9G
// SIG // uwdgR2VgQE6wQuxO7bN2edgKNAltHIAxH+IOVN3lofvl
// SIG // RxCtZJj/UBYufL8FIXrilUEnacOTj5XJjdibIa4NXJzw
// SIG // oq6GaIMMai27dmsAHZat8hZ79haDJLmIz2qoRzEvmtzj
// SIG // cT3XAH5iR9HOiMm4GPoOco3Boz2vAkBq/2mbluIQqBC0
// SIG // N1AI1sM9MIIGBzCCA++gAwIBAgIKYRZoNAAAAAAAHDAN
// SIG // BgkqhkiG9w0BAQUFADBfMRMwEQYKCZImiZPyLGQBGRYD
// SIG // Y29tMRkwFwYKCZImiZPyLGQBGRYJbWljcm9zb2Z0MS0w
// SIG // KwYDVQQDEyRNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
// SIG // ZSBBdXRob3JpdHkwHhcNMDcwNDAzMTI1MzA5WhcNMjEw
// SIG // NDAzMTMwMzA5WjB3MQswCQYDVQQGEwJVUzETMBEGA1UE
// SIG // CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
// SIG // MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSEw
// SIG // HwYDVQQDExhNaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Ew
// SIG // ggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCf
// SIG // oWyx39tIkip8ay4Z4b3i48WZUSNQrc7dGE4kD+7Rp9FM
// SIG // rXQwIBHrB9VUlRVJlBtCkq6YXDAm2gBr6Hu97IkHD/cO
// SIG // BJjwicwfyzMkh53y9GccLPx754gd6udOo6HBI1PKjfpF
// SIG // zwnQXq/QsEIEovmmbJNn1yjcRlOwhtDlKEYuJ6yGT1VS
// SIG // DOQDLPtqkJAwbofzWTCd+n7Wl7PoIZd++NIT8wi3U21S
// SIG // tEWQn0gASkdmEScpZqiX5NMGgUqi+YSnEUcUCYKfhO1V
// SIG // eP4Bmh1QCIUAEDBG7bfeI0a7xC1Un68eeEExd8yb3zuD
// SIG // k6FhArUdDbH895uyAc4iS1T/+QXDwiALAgMBAAGjggGr
// SIG // MIIBpzAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBQj
// SIG // NPjZUkZwCu1A+3b7syuwwzWzDzALBgNVHQ8EBAMCAYYw
// SIG // EAYJKwYBBAGCNxUBBAMCAQAwgZgGA1UdIwSBkDCBjYAU
// SIG // DqyCYEBWJ5flJRP8KuEKU5VZ5KShY6RhMF8xEzARBgoJ
// SIG // kiaJk/IsZAEZFgNjb20xGTAXBgoJkiaJk/IsZAEZFglt
// SIG // aWNyb3NvZnQxLTArBgNVBAMTJE1pY3Jvc29mdCBSb290
// SIG // IENlcnRpZmljYXRlIEF1dGhvcml0eYIQea0WoUqgpa1M
// SIG // c1j0BxMuZTBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8v
// SIG // Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0
// SIG // cy9taWNyb3NvZnRyb290Y2VydC5jcmwwVAYIKwYBBQUH
// SIG // AQEESDBGMEQGCCsGAQUFBzAChjhodHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY3Jvc29mdFJv
// SIG // b3RDZXJ0LmNydDATBgNVHSUEDDAKBggrBgEFBQcDCDAN
// SIG // BgkqhkiG9w0BAQUFAAOCAgEAEJeKw1wDRDbd6bStd9vO
// SIG // eVFNAbEudHFbbQwTq86+e4+4LtQSooxtYrhXAstOIBNQ
// SIG // md16QOJXu69YmhzhHQGGrLt48ovQ7DsB7uK+jwoFyI1I
// SIG // 4vBTFd1Pq5Lk541q1YDB5pTyBi+FA+mRKiQicPv2/OR4
// SIG // mS4N9wficLwYTp2OawpylbihOZxnLcVRDupiXD8WmIsg
// SIG // P+IHGjL5zDFKdjE9K3ILyOpwPf+FChPfwgphjvDXuBfr
// SIG // Tot/xTUrXqO/67x9C0J71FNyIe4wyrt4ZVxbARcKFA7S
// SIG // 2hSY9Ty5ZlizLS/n+YWGzFFW6J1wlGysOUzU9nm/qhh6
// SIG // YinvopspNAZ3GmLJPR5tH4LwC8csu89Ds+X57H2146So
// SIG // dDW4TsVxIxImdgs8UoxxWkZDFLyzs7BNZ8ifQv+AeSGA
// SIG // nhUwZuhCEl4ayJ4iIdBD6Svpu/RIzCzU2DKATCYqSCRf
// SIG // WupW76bemZ3KOm+9gSd0BhHudiG/m4LBJ1S2sWo9iaF2
// SIG // YbRuoROmv6pH8BJv/YoybLL+31HIjCPJZr2dHYcSZAI9
// SIG // La9Zj7jkIeW1sMpjtHhUBdRBLlCslLCleKuzoJZ1GtmS
// SIG // hxN1Ii8yqAhuoFuMJb+g74TKIdbrHk/Jmu5J4PcBZW+J
// SIG // C33Iacjmbuqnl84xKf8OxVtc2E0bodj6L54/LlUWa8kT
// SIG // o/0xggSEMIIEgAIBATCBkDB5MQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQgQ29kZSBTaWdu
// SIG // aW5nIFBDQQITMwAAALARrwqL0Duf3QABAAAAsDAJBgUr
// SIG // DgMCGgUAoIGmMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3
// SIG // AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEV
// SIG // MCMGCSqGSIb3DQEJBDEWBBTQsvIlAGBUzLbhm7eDeDz0
// SIG // xCpfkjBGBgorBgEEAYI3AgEMMTgwNqAcgBoAdABvAG8A
// SIG // bAB3AGkAbgBkAG8AdwAuAGoAc6EWgBRodHRwOi8vbWlj
// SIG // cm9zb2Z0LmNvbTANBgkqhkiG9w0BAQEFAASCAQCii4dr
// SIG // 1fbztqFUS+vIRkMuPFrA7Z7thx6ikg14zDUpI3hx8Y7V
// SIG // h6njtGalJ1m451LT3fWeeQcMXgAwvMG/OQjY9+2GNj9N
// SIG // KndvBzOcp8DRTxfDe+pCOiYsH8lnXQNhKU6qwCehwP43
// SIG // c/L5saR8vxegS0XOFzhgs7aUGgK2T/f1MgitNqmThTSH
// SIG // kkWsIofJbPuk+/wLWog8dFZle1JTtnyL5jnkM/KkEbSY
// SIG // FDZpihgBros8B9SJcH6JrY1qNmn86v0titzWqTBrR9O5
// SIG // K9HNvqIWyB35Yhbk8LmkhdX35oE7o1GHM3PKDhrGWNye
// SIG // 2hR6c0w4xK/qsdy9AH0QjiTvwWf6oYICHzCCAhsGCSqG
// SIG // SIb3DQEJBjGCAgwwggIIAgEBMIGFMHcxCzAJBgNVBAYT
// SIG // AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQH
// SIG // EwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
// SIG // cG9yYXRpb24xITAfBgNVBAMTGE1pY3Jvc29mdCBUaW1l
// SIG // LVN0YW1wIFBDQQIKYQKSSgAAAAAAIDAJBgUrDgMCGgUA
// SIG // oF0wGAYJKoZIhvcNAQkDMQsGCSqGSIb3DQEHATAcBgkq
// SIG // hkiG9w0BCQUxDxcNMTMwMzE1MDYzNDAyWjAjBgkqhkiG
// SIG // 9w0BCQQxFgQUySSj07fJzbb+QaTlPFE+46fgYWIwDQYJ
// SIG // KoZIhvcNAQEFBQAEggEAcQ1mC+qPppWCX6BXM5TO4u9E
// SIG // RmEcg42Zdhww42UuZfdFI6s+1A6gq2RmekXS6mc6cZN8
// SIG // /mR8xCTr2U6H648pcL4mXAeSgrY/Vs58SvD/+TrJxm5v
// SIG // zx+hOYZ9V0Bqen2NXz2lmbYK9IWJ3CoZKEZoVxYAV6BJ
// SIG // isjbo9V0lige+w45eWyIpMcACVCfmy4LdEuv2jTST2S/
// SIG // u4t4mA7MvDQ3qNdWmps4dCgybxPQoN+hMSubIVFkMOm3
// SIG // SeKpM//vVAGhCCmQUMeRfXNgN07UJ8IUwMxDuc1GQ6H8
// SIG // pGIuzQqMMYh25xfMGfa6wP6Pc+u2JeWl+kGwW6uWSGAF
// SIG // FxAXVpv+WQ==
// SIG // End signature block
